# J Stevens
# 19/12/18 REANALYSED 05/04/2024
# Sample analysis of sinusoidal light response curves
# Version 3.5.2 NOW 4.3.3

## Housekeeping ####
rm(list=ls()) # Clear R workspace
#setwd("C:/Users/jsteveg/Dropbox/Basil PoC/Light curves")

library(openxlsx) # good for opening .xlsx files
library(compiler) # rapid calculations
library(tidyverse) # for all the data wrangling functions, not just dplyr and tidyr
library(cowplot) # plotting function
library(lme4) # runs certain types of model 
library(lmerTest) # tests certain types of model
library(emmeans) # provides letter code for figures

enableJIT(2) # Runs compiled code rapidly

## Download data ####
AQ <- NULL
for (f in dir("./Data/Light curves/Sinusoidal wave/Light Responses/", 
              pattern=".xlsx", recursive=T))
{
  print(f) 
  tmp <- read.xlsx(paste("./Data/Light curves/Sinusoidal wave/Light Responses/",f,sep=""),
                   rows=c(11,13:25),sheet=1)
  Ind <- strsplit(f, "_")[[1]]
  Date <- Ind[3]
  Time <- Ind[4]
  Licor <- Ind[5]
  #Ind5 <- strsplit(Ind[5],"\\.")
  Plant <- Ind[7]
  AQ <- rbind(AQ,data.frame(ID=f,Date=Date,Time=Time,Licor=Licor,Plant=Plant, 
                            PAR=tmp$PARi, A=tmp$Photo, gs=tmp$Cond,ETR=tmp$ETR,
                            NPQ=tmp$NPQ,qP=tmp$qP,FvFM=tmp$Fv/tmp$Fm,FvpFmp=tmp[,52],
                            Fs=tmp$Fs, FqpFmp=tmp$PhiPS2,Fmp=tmp[,46]))
}

# Prepare Data
AQ$Time <- as.factor(AQ$Time)
AQ$Time <- factor(AQ$Time,levels(AQ$Time)[c(4,1,2,3)],labels=c("0600","0900","1200","1500"))


# AQ Curve #
#----------#

AQ.A <- AQ %>%
  group_by(Date,Time,Plant) %>%
  slice(2:12) %>%
  group_by(Time,PAR) %>%
  summarise(N=n(),Mean=mean(A),SE=sd(A)/sqrt(N),.groups="drop")


a <- ggplot(AQ.A,aes(x=PAR,y=Mean,ymax=Mean+SE,ymin=Mean-SE,colour=Time))+
  geom_point(size=5,position=position_dodge(width=20))+
  geom_errorbar(position=position_dodge(width=20))+
  ggtitle("A-Q Curve") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

plot(a)

# NPQ vs PAR #
#------------#

# modelling NPQ using Gomperz function
# can be used to check midpoint and max slope of rise

# create function
GOMP <- NULL
AQNPQ <- AQ %>%
  dplyr::select(-FvFM) %>%
  na.omit()

for (id in unique(AQNPQ$ID))
{
  
  print(id)
  AQtmp <- AQNPQ %>%
    filter(ID==id)
  
  gmod <- function(p, PAR)
  {
    gmod <- p[1]*exp(-p[2]*exp(-p[3]*PAR))
    #p[1] = a, p[2] = b, p[3] = c
    # a = max or min value; b slides the curve along the x axis; c = curvature
    
    return(gmod)
  }
  
  opt <- function(p, gobs, PAR) # Mean square minimising function
  {
    return(mean((gmod(p, PAR)-gobs)^2))
  }
  
  
  init <- c(max(AQtmp$NPQ), 3.5, 0.003)
  out <- optim(init, opt, NULL, gobs=AQtmp$NPQ, PAR=AQtmp$PAR, control=list(maxit=2000))
  p <- out$par
  pto <- NULL
  for(a in 1:length(AQtmp$PAR)) {
    
    ptoa <- (gmod(p,AQtmp$PAR[a]))
    pto <- rbind(pto,ptoa)
  }
  AQtmp$model <- pto
  output <- ggplot(AQtmp,aes(y=NPQ,x=PAR, size=4, main=id)) +
    geom_point() +
    geom_line(aes(y=model,x=PAR),colour="red",linewidth=2)+
    ggtitle(paste(id))
  
  GOMP <- rbind(GOMP, data.frame(Id=AQNPQ$ID[1], Time=AQtmp$Time[1], a=p[1], b=p[2], c=p[3]))
  
  ggsave(paste("./Outputs/A-Q Curves/Graph_Gomperz/",
               id,".png", sep=""))
}

GOMP$Mid <- -log(log(2)/GOMP$b)/GOMP$c # midpoint of NPQ curve
GOMP$Max <- log(GOMP$b/GOMP$c) # max slope of NPQ curve

GOM <- GOMP %>%
  group_by(Time) %>%
  summarise(N=n(),Amean=mean(a),Bmean=mean(b),
            Cmean=mean(c),Midmean=mean(Mid),Maxmean=mean(Max))

AQ.NPQ <- AQ %>%
  group_by(Date,Time,Plant) %>%
  slice(2:12) %>%
  group_by(Time,PAR) %>%
  summarise(N=n(),Mean=mean(NPQ),SE=sd(NPQ)/sqrt(N),.groups="drop")

am <- merge(cbind(AQ.NPQ, X=rownames(AQ.NPQ)), cbind(GOM, variable=rownames(GOM)))
# merge both df's

am$Gomperz <- am$Amean*exp(-am$Bmean*exp(-am$Cmean*am$PAR))

b <- ggplot(am,aes(x=PAR,y=Mean,ymax=Mean+SE,ymin=Mean-SE,colour=Time))+
  geom_point(size=2,position=position_dodge(width=20))+
  geom_errorbar(position=position_dodge(width=20))+
  geom_line(aes(x=PAR,y=Gomperz))+
  ggtitle("NPQ-Q Curve")+
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))+
  ylab("NPQ")

plot(b)

AQ.qP <- AQ %>%
  group_by(Date,Time,Plant) %>%
  slice(2:12) %>%
  group_by(Time,PAR) %>%
  summarise(N=n(),Mean=mean(qP),SE=sd(qP)/sqrt(N))

c <- ggplot(AQ.qP,aes(x=PAR,y=Mean,ymax=Mean+SE,ymin=Mean-SE,colour=Time))+
  geom_point(size=5,position=position_dodge(width=20))+
  geom_errorbar(position=position_dodge(width=20))+
  ggtitle("qP-Q curve") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

plot(c)

## Fv Fm comps
fv <- AQ %>%
  filter(PAR==0)

fvmod <- lm(FvFM~Time,data=fv)
summary(fvmod)

fvfm <- AQ %>%
  group_by(Date,Time,Plant) %>%
  filter(Date!="2019-12-17" | Time!="1330") %>%
  slice(13) %>%
  filter(!is.na(FvFM)) %>%
  group_by(Time) %>%
  summarise(N=n(),Mean=mean(FvFM),SE=sd(FvFM)/sqrt(N))

d <- ggplot(fvfm,aes(x=Time,y=Mean,ymax=Mean+SE,ymin=Mean-SE,fill=Time))+
  geom_col()+
  geom_errorbar()+
  ggtitle("Fv / Fm")+
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))+
  ylim(0,1)

plot(d)

# NPQ analysis
mod.mid <- lm(Mid~Time,data=GOMP)
summary(mod.mid)

mod.max <- lm(Max~Time,data=GOMP)
summary(mod.max) 

GOMP1 <- GOMP %>%
  group_by(Time) %>%
  summarise(N=n(),Midmean=mean(Mid),MidSE=sd(Mid)/sqrt(N),Maxmean=mean(Max),MaxSE=sd(Max)/sqrt(N))

e <- ggplot(GOMP1,aes(x=Time,y=Midmean,ymin=Midmean-MidSE,ymax=Midmean+MidSE,fill=Time))+
  geom_col()+
  geom_errorbar()+
  ylab(expression(paste("Midpoint of curve ( PAR, ",~mu,mol,~m^-2,~s^-1," )")))+
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))
plot(e)

g <- ggplot(GOMP1,aes(x=Time,y=Maxmean,ymin=Maxmean-MaxSE,ymax=Maxmean+MaxSE,fill=Time))+
  geom_col()+
  geom_errorbar()+
  ylab(expression(paste("Max slope of curve ( ",~mu,mol,~m^-2,~s^-2," )")))+
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))
plot(g)

# Fv' / Fm' analysis

fvmp <- AQ %>%
  filter(PAR!=0) %>%
  na.omit()

fvmpmod <- lm(FvpFmp~Time*as.factor(PAR),data=fvmp)
summary(fvmpmod)

fvmp.A <- AQ %>%
  group_by(Date,Time,Plant) %>%
  slice(2:12) %>%
  group_by(Time,PAR) %>%
  summarise(N=n(),Mean=mean(FvpFmp),SE=sd(FvpFmp)/sqrt(N))


h <- ggplot(fvmp.A,aes(x=PAR,y=Mean,ymax=Mean+SE,ymin=Mean-SE,colour=Time))+
  geom_point(size=5,position=position_dodge(width=20))+
  geom_errorbar(position=position_dodge(width=20))+
  ggtitle("Fv' / Fm' ") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black")) +
  ylim(0,1)

plot(h)

# Fq' / Fm' analysis
fqmp <- AQ %>%
  filter(PAR!=0) %>%
  na.omit() #%>%
  mutate(Fqmp=(Fmp-Fs)/Fmp)

fqmpmod <- lm(FqpFmp~Time*as.factor(PAR),data=fqmp)
summary(fqmpmod)

fqmp.A <- fqmp %>%
  group_by(Date,Time,Plant) %>%
  slice(2:12) %>%
  group_by(Time,PAR) %>%
  summarise(N=n(),Mean=mean(FqpFmp),SE=sd(FqpFmp)/sqrt(N),.groups="drop")


i <- ggplot(fqmp.A,aes(x=PAR,y=Mean,ymax=Mean+SE,ymin=Mean-SE,colour=Time))+
  geom_point(size=5,position=position_dodge(width=20))+
  geom_errorbar(position=position_dodge(width=20))+
  ggtitle("Fq' / Fm' ") +
  theme(panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
        panel.background = element_blank(), axis.line = element_line(colour = "black"))

plot(i)

## We now model the Fq' / Fm' as a gomperz curve, overlay it on the combined
## data points and create a model curve.
## Then the SP, max and min PAR values entered on the graph

# modelling Fq' / Fm' using Gomperz function

# create function
PHI2 <- NULL
AQPHI2 <- AQ %>%
  dplyr::select(-FvFM) %>%
  na.omit()

for (id in unique(AQPHI2$ID))
{
  
  print(id) 
  AQtmp <- AQPHI2 %>%
    filter(ID==id)
  
  gmod <- function(p, PAR)
  {
    gmod <- p[1]*exp(-p[2]*exp(-p[3]*PAR))
    #p[1] = a, p[2] = b, p[3] = c
    # a = max or min value; b slides the curve along the x axis; c = curvature
    
    return(gmod)
  }
  
  opt <- function(p, gobs, PAR) # Mean square minimising function
  {
    return(mean((gmod(p, PAR)-gobs)^2))
  }
  
  
  init <- c(min(AQtmp$FqpFmp), -3.5, 0.003)
  out <- optim(init, opt, NULL, gobs=AQtmp$FqpFmp, 
               PAR=AQtmp$PAR, control=list(maxit=2000))
  p <- out$par
  pto <- NULL
  for(a in 1:length(AQtmp$PAR)) {
    
    ptoa <- (gmod(p,AQtmp$PAR[a]))
    pto <- rbind(pto,ptoa)
  }
  AQtmp$model <- pto
  output <- ggplot(AQtmp,aes(y=FqpFmp,x=PAR, linewidth=4, main=id)) +
    geom_point() +
    geom_line(aes(y=model,x=PAR),colour="red",linewidth=2)+
    ggtitle(paste(id))
  
  PHI2 <- rbind(PHI2, data.frame(Id=AQPHI2$ID[1], Time=AQtmp$Time[1], 
                                 a=p[1], b=p[2], c=p[3]))
  
  ggsave(paste("./Outputs/A-Q Curves/Graph_Gomperz_Phi2/",
               id,".png", sep=""))
}

PHI2$Mid <- log(log(2)/-PHI2$b)/PHI2$c # midpoint of NPQ curve
PHI2$Max <- log(-PHI2$b/PHI2$c) # max slope of NPQ curve

PHI <- PHI2 %>%
  group_by(Time) %>%
  summarise(N=n(),Amean=mean(a),Bmean=mean(b),
            Cmean=mean(c),Midmean=mean(Mid),Maxmean=mean(Max))

AQ.PHI <- AQ %>%
  group_by(Date,Time,Plant) %>%
  slice(2:12) %>%
  group_by(Time,PAR) %>%
  summarise(N=n(),Mean=mean(FqpFmp),SE=sd(FqpFmp)/sqrt(N),.groups="drop")

qm <- merge(cbind(AQ.PHI, X=rownames(AQ.PHI)), cbind(PHI, variable=rownames(PHI)))
# merge both df's

qm$Gomperz <- qm$Amean*exp(-qm$Bmean*exp(-qm$Cmean*qm$PAR))
qm1 <- qm %>%
  group_by(PAR) %>%
  filter(Time!=1200) %>%
  reframe(fqm=mean(Mean),fqsd=sqrt(sd(SE)^2*(N-1)/(N-1)),Gm=mean(Gomperz),
            a_mean=mean(Amean),b_mean=mean(Bmean),c_mean=mean(Cmean)) %>%
  group_by(PAR) %>%
  summarise(fqm=mean(fqm),fqsd=mean(fqsd),Gm=mean(Gm),a_mean=mean(a_mean),
            b_mean=mean(b_mean),c_mean=mean(c_mean))

## Create a plot
a=unique(qm1$a_mean)
b=unique(qm1$b_mean)
c=unique(qm1$c_mean)
gomfun <- function(x){a*exp(-b*exp(-c*x))-0.04}

fqplot <- ggplot(qm,aes(x=PAR,y=Mean))+
  geom_point(position="jitter",size=1.5)+
  stat_function(fun=gomfun)+
  geom_segment(x=450,y=-0.08,yend=gomfun(450),linetype=2)+
  geom_segment(x=100,y=-0.08,yend=gomfun(100),linetype=2)+
  geom_segment(x=-85,xend=450,y=gomfun(450),linetype=2)+
  geom_segment(x=-85,xend=100,y=gomfun(100),linetype=2)+
  geom_segment(x=-85,xend=245,y=gomfun(250),colour="red")+
  annotate("text",x=110,y=0.1,label = "PFD =\n 100",hjust=0,size=3)+
  annotate("text",x=-60,y=0.555,label = "Setpoint \n0.52",hjust=0,size=3)+
  annotate("text",x=460,y=0.1,label = "PFD =\n 450",hjust=0,size=3)+
  theme_bw()+
  theme(panel.grid = element_blank(),
        panel.border = element_rect(color = "black", 
                                    fill = NA, 
                                    linewidth = 1.5),
        axis.title=element_text(size=11))+
  xlab(expression(paste("PFD ( ",~mu,mol,~m^-2,s^-1," )"))) +
  ylab(expression(paste(italic("Fq'/Fm' "), " (a.u.)"))) +
  coord_cartesian(clip = 'on', ylim=c(-0, 0.8),xlim=c(-20,1320))+
  scale_x_continuous(breaks = c(0,250,500,750,1000,1250))

fqplot

ggsave(file="./Outputs/Setpoint_analysis.png",fqplot)
#ggsave(file="./Outputs/Final Plots/Setpoint_analysis.png")

save(file="./Outputs/Setpoint_analysis.rdata",fqplot)
